#!/usr/bin/env python
# coding=utf-8
#############################################################################
# Copyright (c): 2019, Huawei Tech. Co., Ltd. 2019. All rights reserved.
# script for gds_ctl
# Version      : 1.0.0
# change log:
#############################################################################
import socket
import os
import sys
import subprocess
import signal
import re

try:
    import xml.etree.cElementTree as ET
except ImportError:
    import xml.etree.ElementTree as ET

#Exit Code
EXIT_SUCCESS = 0
EXIT_FAILURE = -1
EXIT_STATUS_IDEL = 1
EXIT_STATUS_RUNNING = 2

class gds_entry:
    port = -1
    name = None
    log_file = None
    data_dir = None
    err_dir = None
    data_seg = None
    err_seg = None
    ctl_file = None
    recursive = False
    daemon = False
    host = None
    ip = None
    parallel = -1
    help = False

    def is_valid(self):
        if self.help == True:
            return True
        if self.port < 1024 or self.port > 65535:
            return False
        if self.data_dir == None:
            return False
        if self.name == None:
            return False
        if re.search('^\w+([-+.]\w+)*$', self.name) == None:
            return False

        return True

    def to_str(self):
        str = ('name:{0}'.format(self.name))
        if self.ip != None:
            str += (' ip:{0}'.format(self.ip))

        str += (' port:{0:d}'.format(self.port))
        str += (' data_dir:{0}'.format(self.dta_dir))
        if self.err_dir != None:
            str += (' err_dir:{0}'.format(self.drr_dir))

        if self.data_seg != None:
            str += (' data_seg:{0}'.format(self.data_seg))

        if self.err_seg != None:
            str += (' err_seg:{0}'.format(self.err_seg))

        if self.log_file != None:
            str += (' log_file:{0}'.format(self.log_file))

        if self.host != None:
            str += (' host:{0}'.format(self.host))

        if self.recursive == True:
            str += ' recursive:{true}'
        else:
            str += ' recursive:{false}'
        
        if self.daemon == True:
            str += ' daemon:{true}'
        else:
            str += ' daemon:{false}'

        if self.enablessl == True:
            str += ' enablessl:{true}'
        else:
            str += ' enablessl:{false}'

        if self.parallel != None:
            str += (' parallel:{0}'.format(self.parallel))
        if self.help == True:
            str += ' help:{true}'
        else:
            str += ' help:{false}'

        return str

def usage():
    print "gds_ctl.py [start|stop|status]"

#Read the GDS configration
def read_conf(filename):
    try:
        tree = ET.ElementTree(file=filename)
        root = tree.getroot()
        gds_array = []
        gds_dict = {}

    except:
            print 'Invalid gds configuration file \"' + filename + '\"'
            sys.exit(EXIT_FAILURE)
    for cell in root:

        entry = gds_entry()
        entry.port = int(cell.get('port', '-1'))
        entry.data_dir = cell.get('data_dir', None)
        entry.err_dir = cell.get('err_dir', None)
        entry.data_seg = cell.get('data_seg', None)
        entry.err_seg = cell.get('err_seg', None)
        entry.recursive = (cell.get('recursive', 'False').lower() == 'true')
        entry.daemon = (cell.get('daemon', 'False').lower() == 'true')
        entry.help = (cell.get('help', 'False').lower() == 'true')
        #make the data_dir as a ABS path
        if entry.data_dir != None:
            entry.data_dir = entry.data_dir.strip()
            if os.path.isabs(entry.data_dir) == False:
                entry.data_dir = "%s/%s" % (sys.path[0], entry.data_dir)

        #make the err_dir as a ABS path
        if entry.err_dir != None:
            entry.err_dir = entry.err_dir.strip()
            if os.path.isabs(entry.err_dir) == False:
                entry.err_dir = "%s/%s" % (sys.path[0], entry.err_dir)

        entry.name = cell.get('name', None)
        entry.ctl_file = "%s%s" % ('.gds.', entry.name)
        entry.log_file = cell.get('log_file', None)
        entry.host = cell.get('host', None)
        entry.ip = cell.get('ip', None)
        entry.parallel = cell.get('parallel', None)

        if entry.is_valid() != True:
            print "%s%s%s" %('Invalid item in configuration file \"', filename, '\"')
            sys.exit(EXIT_FAILURE);
        gds_array.append(entry)
        if gds_dict.has_key(entry.name):
            print ('Conflict GDS name \"{0}\"'.format(entry.name))
            sys.exit(EXIT_FAILURE)
        gds_dict[entry.name] = entry

    gds_dict.clear()
    return gds_array

#Start GDS
def gds_start(entries):
    for cell in entries:
        if cell.help == True:
            cmd = "%s%s" % (g_bin_path, '/gds')
            if subprocess.call([cmd, "-h"], shell = False) !=0:
                return EXIT_FAILURE
            else:
                return EXIT_SUCCESS

        if __is_GDS_exist__(cell):
            print (('GDS {0} already exists.').format(cell.name))
            return EXIT_FAILURE
        #Create the data directory if dosen't exist,
        if os.path.exists(cell.data_dir) == False:
            os.makedirs(cell.data_dir)

        #Create the error log directory if dosen't exist,
        if os.path.exists(cell.err_dir) == False:
            os.makedirs(cell.err_dir)

        if os.path.exists(cell.ctl_file):
            subprocess.call(['rm', cell.ctl_file], shell = False)

        cmd = "%s%s" % (g_bin_path, '/gds')
        cmd_list = [cmd, "-d", cell.data_dir, "-s", cell.ctl_file, "-D"]
        if cell.ip != None:
            cmd_list.append('-p')
            cmd_list.append(('{0}:{1:d}'.format(cell.ip, cell.port)))
        else:
            cmd_list.append('-p')
            cmd_list.append(str(cell.port))
        if cell.log_file != None:
            cmd_list.append("-l")
            cmd_list.append(cell.log_file)
        if cell.recursive == True:
            cmd_list.append("-r")
        if cell.daemon == True:
            cmd_list.append("-D")
        if cell.host != None:
            cmd_list.append("-H")
            cmd_list.append(cell.host)
        if cell.parallel != -1:
            cmd_list.append("-t")
            cmd_list.append(cell.parallel)
        if cell.err_dir != None:
           cmd_list.append("-e")
           cmd_list.append(cell.err_dir)
        if cell.data_seg != None:
            cmd_list.append("-S")
            cmd_list.append(cell.data_seg)
        if cell.err_seg != None:
            cmd_list.append("-E")
            cmd_list.append(cell.data_seg)
        cmd_list.append("--enable-ssl")
        cmd_list.append("off")
        if subprocess.call(cmd_list, shell = False) != 0:
            print ('Start GDS {0:20s}[ERROR]'.format(cell.name))
            return EXIT_FAILURE
        print ('Start GDS {0:30s}[OK]'.format(cell.name))
    return EXIT_SUCCESS

#Stop the GDS
def gds_stop_entries(entries):
    gds_proc = []

    out,err = subprocess.Popen(['ps', 'ax'], stdout = subprocess.PIPE).communicate()
    for line in out.splitlines():
        if 'gds' in line:
            gds_proc.append(line)

    for cell in entries:
        for proc in gds_proc:
            if cell.ctl_file in proc.split(' '):
                pid = int(proc.split(None, 1)[0])
                os.kill(pid, signal.SIGRTMIN)
                if (os.path.exists(cell.ctl_file)):
                    subprocess.call(['rm', cell.ctl_file], shell=False)
                if __is_GDS_exist__(cell) == False:
                    print ('Stop GDS {0:32s}[OK]'.format(cell.name))
                else:
                    print ('Stop GDS {0:32s}[ERROR]'.format(cell.name))
    return EXIT_SUCCESS

def gds_stop_all():
    out,err = subprocess.Popen(['ps', 'ax'], stdout = subprocess.PIPE).communicate()

    for line in out.splitlines():
        if 'gds' in line and '-p' in line and '-d' in line:
            tokens = line.split(' ')
            i = 0
            for token in tokens:
                if token == '-p':
                    break
                i = i + 1
            single = tokens[i+1]
            pid = int(line.split(None, 1)[0])
            os.kill(pid, signal.SIGRTMIN)
            if __is_GDS_exist__(pid) == False:
                print ('Stop GDS {0:32s}[OK]'.format(single))
            else:
                print ('Stop GDS {0:32s}[ERROR]'.format(single))
    return EXIT_SUCCESS

def gds_stop_single(single):
    out,err = subprocess.Popen(['ps', 'ax'], stdout = subprocess.PIPE).communicate()
    for line in out.splitlines():
        if 'gds' in line and '-p' in line and '-d' in line:
            tokens = line.split(' ')
            i = 0
            for token in tokens:
                if token == '-p':
                    break
                i = i + 1
            if tokens[i+1] != single:
                continue
            pid = int(line.split(None, 1)[0])
            os.kill(pid, signal.SIGRTMIN)
            if __is_GDS_exist__(pid) == False:
                print ('Stop GDS {0:32s}[OK]'.format(single))
            else:
                print ('Stop GDS {0:32s}[ERROR]'.format(single))
    return EXIT_SUCCESS

def __is_GDS_exist__(entry):
    out,err = subprocess.Popen(['ps', 'ax'], stdout = subprocess.PIPE, shell = True).communicate()
    for line in out.splitlines():
        if ('gds' in line) and (entry.ctl_file in line.split(' ')):
            return True
    return False

def __is_GDS_exist__(pid):
    out,err = subprocess.Popen(['ps', 'ax'], stdout = subprocess.PIPE, shell = True).communicate()
    for line in out.splitlines():
        if ('gds' in line) and (pid in line.split(' ')):
            return True
    return False

#Notice: Now the 'STATUS' command can only return the status of the first GDS in configuration file.
def gds_status(self):
    gds = self[0]
    if gds == None:
        return EXIT_FAILURE;
    if os.path.exists(gds.ctl_file) == False:
        return EXIT_FAILURE
    try:
        client = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        client.settimeout(5)
        client.connect(gds.ctl_file)
        client.send("Q")
        result = client.recv(1)
    except socket.timeout:
        client.close()
        return EXIT_FAILURE
    except socket.error:
        return EXIT_FAILURE
    if result == 'I':
        print (('GDS {0:32}[IDEL]').format(gds.name))
        return EXIT_STATUS_IDEL
    elif result == 'R':
        print (('GDS {0:32}[RUNNING]').format(gds.name))
        return EXIT_STATUS_RUNNING
    else:
        return EXIT_FAILURE

def main():
    global g_cfg_path;
    global g_bin_path;

    #The configuration file gds.cfg is under '$install_dir/config/' by default.
    g_cfg_path = sys.path[0] + '/config/gds.conf'
    g_bin_path = sys.path[0]

    #Verify if the configuration file exists
    if os.path.exists(g_cfg_path) == False :
        g_cfg_path = sys.path[0] + '/gds.conf'
        if os.path.exists(g_cfg_path) == False :
            print "Can't find GDS configure file " + g_cfg_path
            sys.exit(EXIT_FAILURE)
    if len(sys.argv) == 1:
        usage()
        sys.exit(EXIT_FAILURE)
    cmd = sys.argv[1]
    if str(cmd).lower() == 'start':
        sys.exit(gds_start(read_conf(g_cfg_path)))
    elif str(cmd).lower() == 'stop':
        #stop gds entries
        if len(sys.argv) == 2:
            sys.exit(gds_stop_entries(read_conf(g_cfg_path)))
        if len(sys.argv) == 3:
            cmd = sys.argv[2]
            #stop all gds which could be stopped
            if cmd == "all":
                sys.exit(gds_stop_all())
            #stop single gds specified by "[ip]:port"
            else:
                sys.exit(gds_stop_single(cmd))
    elif str(cmd).lower() == 'status':
        sys.exit(gds_status(read_conf(g_cfg_path)))
    else:
        print "Invalid commands \"" + cmd + "\""
        usage()
        sys.exit(EXIT_FAILURE)

if __name__ == "__main__":
    main()
